<?php
// --- SERVER SIDE LOGIC ---
$dataFile = 'sm7_data.json';

// Handle Data Updates (From Mobile)
if (isset($_POST['update'])) {
    $data = $_POST['data'];
    file_put_contents($dataFile, $data);
    echo "Saved";
    exit;
}

// Handle Data Reading (From Overlay)
if (isset($_GET['read'])) {
    if (file_exists($dataFile)) {
        header('Content-Type: application/json');
        readfile($dataFile);
    } else {
        echo "{}";
    }
    exit;
}

// Check which view to load
$view = isset($_GET['view']) ? $_GET['view'] : 'admin';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SM7 Pickleball</title>
    <style>
        :root {
            --primary: #0056b3;
            --accent: #ffd700;
            --dark: #1a1a1a;
            --light: #f4f4f9;
        }
        body { font-family: sans-serif; margin: 0; overflow: hidden; background: var(--light); }

        /* --- OVERLAY STYLES (vMix) --- */
        <?php if ($view == 'overlay'): ?>
        body { background: transparent; }
        #app {
            display: flex; height: 100vh; width: 100vw;
            align-items: flex-end; justify-content: center; padding-bottom: 50px;
        }
        .scoreboard {
            display: flex; width: 900px; height: 110px;
            background: rgba(0,0,0,0.85); color: white;
            border: 2px solid white; border-radius: 12px; overflow: hidden;
            box-shadow: 0 4px 20px rgba(0,0,0,0.5);
        }
        .team-box {
            flex: 2; display: flex; flex-direction: column;
            justify-content: center; align-items: center; position: relative;
        }
        .team-name { font-size: 24px; text-transform: uppercase; font-weight: bold; }
        .team-score { font-size: 55px; font-weight: 800; color: var(--accent); line-height: 1; }
        .info-box {
            flex: 1.2; background: #333; display: flex; flex-direction: column;
            justify-content: center; align-items: center; border-left: 1px solid #555; border-right: 1px solid #555;
        }
        .server-num { font-size: 40px; font-weight: bold; }
        .server-label { font-size: 10px; text-transform: uppercase; color: #aaa; letter-spacing: 1px; }
        .serving-dot {
            width: 12px; height: 12px; background: #0f0; border-radius: 50%;
            box-shadow: 0 0 8px #0f0; position: absolute; top: 15px; right: 15px;
            display: none;
        }
        .active-serve .serving-dot { display: block; }

        /* --- ADMIN STYLES (Mobile) --- */
        <?php else: ?>
        #app { padding: 10px; height: 100vh; box-sizing: border-box; display: flex; flex-direction: column; gap: 10px; overflow-y: auto; }
        .header { text-align: center; margin-bottom: 5px; }
        .inputs { display: flex; gap: 5px; margin-bottom: 10px; }
        input { flex: 1; padding: 10px; border: 1px solid #ccc; border-radius: 5px; text-align: center; font-size: 14px;}
        
        .score-area { display: flex; gap: 10px; flex: 1; min-height: 200px; }
        .team-col { flex: 1; display: flex; flex-direction: column; gap: 10px; background: white; padding: 10px; border-radius: 10px; box-shadow: 0 2px 5px rgba(0,0,0,0.1); }
        
        h3 { margin: 0; text-align: center; font-size: 16px; color: #555; overflow: hidden; white-space: nowrap; text-overflow: ellipsis; }
        .score-display { font-size: 50px; font-weight: bold; text-align: center; color: var(--primary); margin: 10px 0; }
        
        button { border: none; border-radius: 8px; font-weight: bold; cursor: pointer; touch-action: manipulation; }
        button:active { opacity: 0.7; transform: scale(0.98); }
        
        .btn-add { flex: 2; background: #28a745; color: white; font-size: 24px; }
        .btn-sub { flex: 0.5; background: #dc3545; color: white; font-size: 18px; }
        .btn-serve { padding: 10px; background: #ddd; color: #333; }
        .btn-serve.active { background: var(--primary); color: white; }
        
        .middle-controls { display: flex; gap: 10px; margin-top: 5px; }
        .btn-sideout { flex: 2; background: var(--accent); color: #000; padding: 15px; font-size: 18px; }
        .btn-reset { flex: 1; background: #6c757d; color: white; font-size: 14px; }
        
        .server-toggle { background: #333; color: white; padding: 15px; border-radius: 8px; text-align: center; font-size: 18px; margin-bottom: 5px;}
        <?php endif; ?>
    </style>
</head>
<body>

    <div id="app">
        <?php if ($view == 'overlay'): ?>
            <div class="scoreboard">
                <div class="team-box" id="ov-t1">
                    <div class="serving-dot"></div>
                    <div class="team-name" id="disp-name1">TEAM A</div>
                    <div class="team-score" id="disp-score1">0</div>
                </div>
                <div class="info-box">
                    <div class="server-label" id="disp-game">FINAL</div>
                    <div class="server-num" id="disp-serv">2</div>
                    <div class="server-label">SERVER</div>
                </div>
                <div class="team-box" id="ov-t2">
                    <div class="serving-dot"></div>
                    <div class="team-name" id="disp-name2">TEAM B</div>
                    <div class="team-score" id="disp-score2">0</div>
                </div>
            </div>

        <?php else: ?>
            <div class="header"><b>SM7 Pickleball Scorer</b></div>
            
            <div class="inputs">
                <input type="text" id="name1" placeholder="Team 1" oninput="save()" value="Team A">
                <input type="text" id="game" placeholder="Match Info" oninput="save()" value="Final">
                <input type="text" id="name2" placeholder="Team 2" oninput="save()" value="Team B">
            </div>

            <div class="server-toggle" onclick="toggleServer()">
                Server: <span id="serv-disp" style="font-weight:bold; color: var(--accent);">2</span>
                <div style="font-size: 12px; color: #aaa; margin-top: 2px;">(Tap to switch 1 / 2)</div>
            </div>

            <div class="score-area">
                <div class="team-col">
                    <button class="btn-serve" id="btn-serve1" onclick="setServe(1)">Serve Here</button>
                    <h3 id="lbl-name1">Team A</h3>
                    <div class="score-display" id="val-score1">0</div>
                    <button class="btn-add" onclick="modScore(1, 1)">+1</button>
                    <button class="btn-sub" onclick="modScore(1, -1)">-1</button>
                </div>

                <div class="team-col">
                    <button class="btn-serve" id="btn-serve2" onclick="setServe(2)">Serve Here</button>
                    <h3 id="lbl-name2">Team B</h3>
                    <div class="score-display" id="val-score2">0</div>
                    <button class="btn-add" onclick="modScore(2, 1)">+1</button>
                    <button class="btn-sub" onclick="modScore(2, -1)">-1</button>
                </div>
            </div>

            <div class="middle-controls">
                <button class="btn-reset" onclick="resetGame()">RESET</button>
                <button class="btn-sideout" onclick="sideOut()">SIDE OUT ⇄</button>
            </div>
            <div id="status" style="text-align:center; font-size:12px; color:#aaa; margin-top:5px;">Ready</div>
        <?php endif; ?>
    </div>

    <script>
        // --- JAVASCRIPT LOGIC ---
        const isOverlay = <?php echo ($view == 'overlay' ? 'true' : 'false'); ?>;
        
        // Default State
        let state = {
            n1: "Team A", n2: "Team B", game: "Final",
            s1: 0, s2: 0,
            servNum: 2, servTeam: 1
        };

        // --- OVERLAY MODE ---
        if (isOverlay) {
            function pollData() {
                fetch('?read=1&t=' + Date.now()) // timestamp prevents caching
                    .then(r => r.json())
                    .then(data => {
                        if(data.n1) updateOverlay(data);
                    });
            }
            
            function updateOverlay(data) {
                document.getElementById('disp-name1').innerText = data.n1;
                document.getElementById('disp-name2').innerText = data.n2;
                document.getElementById('disp-score1').innerText = data.s1;
                document.getElementById('disp-score2').innerText = data.s2;
                document.getElementById('disp-game').innerText = data.game;
                document.getElementById('disp-serv').innerText = data.servNum;
                
                document.getElementById('ov-t1').classList.remove('active-serve');
                document.getElementById('ov-t2').classList.remove('active-serve');
                
                if(data.servTeam == 1) document.getElementById('ov-t1').classList.add('active-serve');
                else document.getElementById('ov-t2').classList.add('active-serve');
            }

            setInterval(pollData, 1000); // Check for updates every 1 second
        }

        // --- ADMIN MODE ---
        else {
            function save() {
                // Update text inputs to state
                state.n1 = document.getElementById('name1').value;
                state.n2 = document.getElementById('name2').value;
                state.game = document.getElementById('game').value;
                
                // Update UI Labels locally immediately for snappiness
                document.getElementById('lbl-name1').innerText = state.n1;
                document.getElementById('lbl-name2').innerText = state.n2;
                document.getElementById('val-score1').innerText = state.s1;
                document.getElementById('val-score2').innerText = state.s2;
                document.getElementById('serv-disp').innerText = state.servNum;
                
                document.getElementById('btn-serve1').className = state.servTeam == 1 ? 'btn-serve active' : 'btn-serve';
                document.getElementById('btn-serve2').className = state.servTeam == 2 ? 'btn-serve active' : 'btn-serve';

                // Send to Server
                let fd = new FormData();
                fd.append('update', '1');
                fd.append('data', JSON.stringify(state));
                
                document.getElementById('status').innerText = "Syncing...";
                fetch('', { method: 'POST', body: fd })
                    .then(r => r.text())
                    .then(() => document.getElementById('status').innerText = "Live");
            }

            function modScore(team, amount) {
                if (team == 1) state.s1 = Math.max(0, state.s1 + amount);
                if (team == 2) state.s2 = Math.max(0, state.s2 + amount);
                save();
            }

            function setServe(team) {
                state.servTeam = team;
                save();
            }

            function toggleServer() {
                state.servNum = state.servNum == 1 ? 2 : 1;
                save();
            }

            function sideOut() {
                state.servTeam = state.servTeam == 1 ? 2 : 1;
                state.servNum = 1; // Reset to first server on sideout
                save();
            }

            function resetGame() {
                if(confirm("Reset scores to 0-0?")) {
                    state.s1 = 0; state.s2 = 0;
                    state.servNum = 2; state.servTeam = 1;
                    save();
                }
            }
            
            // Init
            save();
        }
    </script>
</body>
</html>