<?php
// --- SM7 TOURNAMENT MANAGER (Single File) ---
$dataFile = 'sm7_db.json';

// --- INITIALIZE DATABASE ---
if (!file_exists($dataFile)) {
    $initialDB = [
        'teams' => [], // id, name, p1, p2, logo
        'courts' => [] // The 4 active courts
    ];
    // Init 4 courts
    for($i=1; $i<=4; $i++) $initialDB['courts'][$i] = [
        'id' => $i, 'n1'=>'', 'n2'=>'', 's1'=>0, 's2'=>0, 
        'l1'=>'', 'l2'=>'', 'game'=>'Match', 'serv'=>2, 'team'=>1
    ];
    file_put_contents($dataFile, json_encode($initialDB));
}

// --- API HANDLERS ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $db = json_decode(file_get_contents($dataFile), true);
    $action = $_POST['action'];

    if ($action === 'save_team') {
        $id = $_POST['id'] ?: uniqid();
        $team = [
            'id' => $id,
            'name' => $_POST['name'],
            'p1' => $_POST['p1'],
            'p2' => $_POST['p2'],
            'logo' => $_POST['logo']
        ];
        $db['teams'][$id] = $team;
    }
    
    if ($action === 'delete_team') {
        unset($db['teams'][$_POST['id']]);
    }

    if ($action === 'dispatch_match') {
        // Send specific teams to a court
        $cID = $_POST['court_id'];
        $t1 = $db['teams'][$_POST['t1_id']];
        $t2 = $db['teams'][$_POST['t2_id']];
        
        $db['courts'][$cID] = [
            'id' => $cID,
            'n1' => $t1['name'], 'n2' => $t2['name'],
            'l1' => $t1['logo'], 'l2' => $t2['logo'],
            'p1' => $t1['p1']." / ".$t1['p2'], // Combine player names
            'p2' => $t2['p1']." / ".$t2['p2'],
            's1' => 0, 's2' => 0, 'serv' => 2, 'team' => 1,
            'game' => $_POST['match_title']
        ];
    }

    if ($action === 'update_score') {
        $cID = $_POST['court'];
        $db['courts'][$cID] = json_decode($_POST['data'], true);
    }

    file_put_contents($dataFile, json_encode($db));
    echo json_encode(['status'=>'ok']);
    exit;
}

if (isset($_GET['api_read'])) {
    header('Content-Type: application/json');
    readfile($dataFile);
    exit;
}

// --- ROUTING VIEWS ---
$view = $_GET['view'] ?? 'admin'; // admin, scorer, overlay, public
$courtID = $_GET['court'] ?? 1;
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SM7 Manager</title>
    <script src="https://unpkg.com/@phosphor-icons/web"></script>
    <style>
        :root {
            --nav-bg: #0f172a; --nav-text: #94a3b8; --nav-active: #ffffff;
            --accent: #bef264; /* Pickleball Green */
            --bg: #f1f5f9; --card: #ffffff; --border: #e2e8f0;
            --text: #334155; --dark: #1e293b;
            --blue: #2563eb; --red: #ef4444;
        }
        body { font-family: 'Inter', system-ui, sans-serif; margin: 0; background: var(--bg); color: var(--text); height: 100vh; overflow: hidden; }
        
        /* === ADMIN DASHBOARD LAYOUT === */
        .admin-layout { display: flex; height: 100%; }
        .sidebar { width: 250px; background: var(--nav-bg); display: flex; flex-direction: column; padding: 20px; flex-shrink: 0; }
        .brand { color: white; font-weight: 800; font-size: 24px; margin-bottom: 30px; display: flex; align-items: center; gap: 10px; }
        .brand span { color: var(--accent); }
        
        .nav-item { 
            padding: 12px 15px; color: var(--nav-text); text-decoration: none; border-radius: 8px; 
            margin-bottom: 5px; display: flex; align-items: center; gap: 10px; font-weight: 500; transition: 0.2s; cursor: pointer;
        }
        .nav-item:hover, .nav-item.active { background: rgba(255,255,255,0.1); color: var(--nav-active); }
        .nav-item i { font-size: 20px; }

        .main-content { flex: 1; padding: 30px; overflow-y: auto; }
        
        /* Sections */
        .section { display: none; }
        .section.active { display: block; }
        
        h1 { margin: 0 0 20px 0; color: var(--dark); font-size: 28px; }

        /* Cards & Tables */
        .card { background: var(--card); border-radius: 12px; padding: 20px; border: 1px solid var(--border); box-shadow: 0 2px 5px rgba(0,0,0,0.03); margin-bottom: 20px; }
        .grid-courts { display: grid; grid-template-columns: repeat(2, 1fr); gap: 20px; }
        
        .btn { padding: 10px 16px; border: none; border-radius: 6px; font-weight: 600; cursor: pointer; display: inline-flex; align-items: center; gap: 8px; font-size: 14px; }
        .btn-primary { background: var(--nav-bg); color: white; }
        .btn-accent { background: var(--accent); color: var(--dark); }
        .btn-danger { background: #fee2e2; color: var(--red); }
        
        input, select { padding: 10px; border: 1px solid var(--border); border-radius: 6px; width: 100%; box-sizing: border-box; margin-bottom: 10px; }
        label { font-size: 12px; font-weight: bold; color: #64748b; display: block; margin-bottom: 4px; }

        /* Court Monitor */
        .monitor-card { background: white; padding: 15px; border-radius: 10px; border: 1px solid var(--border); display: flex; flex-direction: column; gap: 10px; }
        .m-header { font-weight: bold; color: var(--blue); display: flex; justify-content: space-between; }
        .m-score { font-size: 32px; font-weight: 800; text-align: center; margin: 10px 0; color: var(--dark); }
        .m-teams { font-size: 14px; text-align: center; color: #64748b; }
        .tag { background: #e2e8f0; padding: 4px 8px; border-radius: 4px; font-size: 11px; font-weight: bold; }

        /* Tables */
        table { width: 100%; border-collapse: collapse; }
        th { text-align: left; color: #64748b; font-size: 12px; padding: 10px; border-bottom: 1px solid var(--border); }
        td { padding: 12px 10px; border-bottom: 1px solid var(--border); font-size: 14px; }
        .team-logo-small { width: 24px; height: 24px; border-radius: 50%; object-fit: contain; vertical-align: middle; margin-right: 8px; background: #eee; }

        /* === SCORER & OVERLAY SPECIFIC === */
        <?php if($view == 'scorer'): ?>
        .scorer-body { background: #fff; height: 100vh; display: flex; flex-direction: column; }
        .scr-header { padding: 10px; background: var(--nav-bg); color: white; display: flex; align-items: center; justify-content: space-between; }
        .scr-grid { flex: 1; display: flex; flex-direction: column; padding: 10px; gap: 10px; }
        .scr-team { flex: 1; background: #f8fafc; border: 1px solid var(--border); border-radius: 12px; display: flex; flex-direction: column; align-items: center; justify-content: center; position: relative; }
        .scr-name { font-size: 18px; font-weight: 700; }
        .scr-score { font-size: 64px; font-weight: 800; color: var(--blue); line-height: 1; }
        .scr-srv-btn { margin-top: 10px; padding: 6px 12px; background: #e2e8f0; border-radius: 20px; font-size: 12px; font-weight: bold; color: #64748b; border: none; }
        .scr-srv-btn.active { background: var(--accent); color: var(--dark); box-shadow: 0 0 10px rgba(190, 242, 100, 0.5); }
        .scr-ctrl { display: flex; width: 100%; gap: 10px; padding: 0 20px; box-sizing: border-box; margin-top: 10px; }
        .btn-big { flex: 1; padding: 15px; border-radius: 8px; border: none; font-weight: bold; font-size: 20px; }
        .add { background: #22c55e; color: white; }
        .sub { background: #ef4444; color: white; max-width: 60px; }
        .scr-footer { padding: 10px; display: grid; grid-template-columns: 2fr 1fr; gap: 10px; }
        <?php endif; ?>

        <?php if($view == 'overlay'): ?>
        body { background: transparent; }
        #overlay { display: flex; width: 100vw; height: 100vh; align-items: flex-end; justify-content: center; padding-bottom: 50px; }
        .ov-board { display: flex; width: 900px; height: 120px; background: rgba(15, 23, 42, 0.95); border-radius: 16px; overflow: hidden; border: 2px solid rgba(255,255,255,0.1); box-shadow: 0 10px 30px rgba(0,0,0,0.5); color: white; }
        .ov-team { flex: 1; display: flex; align-items: center; padding: 0 25px; position: relative; }
        .ov-logo { width: 60px; height: 60px; background: white; border-radius: 50%; padding: 4px; object-fit: contain; }
        .ov-info { margin: 0 20px; flex: 1; }
        .ov-name { font-size: 22px; font-weight: 800; text-transform: uppercase; letter-spacing: 0.5px; }
        .ov-players { font-size: 13px; color: #94a3b8; font-weight: 500; margin-top: 2px; }
        .ov-score { font-size: 60px; font-weight: 800; color: var(--accent); }
        .ov-mid { width: 140px; background: rgba(255,255,255,0.05); display: flex; flex-direction: column; align-items: center; justify-content: center; border-left: 1px solid rgba(255,255,255,0.1); border-right: 1px solid rgba(255,255,255,0.1); }
        .ov-srv-num { font-size: 42px; font-weight: 700; line-height: 1; }
        .ov-dot { width: 12px; height: 12px; background: #22c55e; border-radius: 50%; box-shadow: 0 0 10px #22c55e; position: absolute; top: 15px; display: none; }
        .serving .ov-dot { display: block; }
        <?php endif; ?>
    </style>
</head>
<body>

<?php if($view == 'admin'): ?>
<div class="admin-layout">
    <div class="sidebar">
        <div class="brand"><i class="ph ph-trophy"></i> SM7<span>Manager</span></div>
        <div class="nav-item active" onclick="nav('dash')"><i class="ph ph-squares-four"></i> Dashboard</div>
        <div class="nav-item" onclick="nav('teams')"><i class="ph ph-users"></i> Teams & Players</div>
        <div class="nav-item" onclick="nav('schedule')"><i class="ph ph-calendar-check"></i> Dispatcher</div>
        <div class="nav-item" onclick="window.open('?view=public','_blank')"><i class="ph ph-monitor-play"></i> Public View</div>
    </div>
    
    <div class="main-content">
        <div id="sec-dash" class="section active">
            <h1>Live Courts</h1>
            <div class="grid-courts" id="dash-grid"></div>
        </div>

        <div id="sec-teams" class="section">
            <h1>Teams Database</h1>
            <div class="card">
                <div style="display:flex; gap:10px; margin-bottom:20px;">
                    <input type="text" id="new-t-name" placeholder="Team Name (e.g. Titans)">
                    <input type="text" id="new-t-p1" placeholder="Player 1">
                    <input type="text" id="new-t-p2" placeholder="Player 2">
                    <input type="text" id="new-t-logo" placeholder="Logo URL">
                    <button class="btn btn-accent" onclick="addTeam()">+ Add Team</button>
                </div>
                <table>
                    <thead><tr><th>Logo</th><th>Team Name</th><th>Players</th><th>Action</th></tr></thead>
                    <tbody id="team-list"></tbody>
                </table>
            </div>
        </div>

        <div id="sec-schedule" class="section">
            <h1>Match Dispatcher</h1>
            <div class="card" style="max-width: 500px;">
                <label>Select Team 1</label>
                <select id="disp-t1"></select>
                <label>Select Team 2</label>
                <select id="disp-t2"></select>
                <label>Match Title</label>
                <input type="text" id="disp-title" value="Final Match">
                <label>Target Court</label>
                <select id="disp-court">
                    <option value="1">Court 1</option>
                    <option value="2">Court 2</option>
                    <option value="3">Court 3</option>
                    <option value="4">Court 4</option>
                </select>
                <br><br>
                <button class="btn btn-primary" style="width:100%" onclick="dispatchMatch()">🚀 Send to Court</button>
            </div>
        </div>
    </div>
</div>

<script>
    let db = { teams: {}, courts: {} };

    // Navigation
    function nav(id) {
        document.querySelectorAll('.section').forEach(e => e.classList.remove('active'));
        document.getElementById('sec-'+id).classList.add('active');
        document.querySelectorAll('.nav-item').forEach(e => e.classList.remove('active'));
        event.currentTarget.classList.add('active');
    }

    // Data Loading
    function loadDB() {
        fetch('?api_read=1&t='+Date.now()).then(r=>r.json()).then(data => {
            db = data;
            renderDash();
            renderTeams();
            renderDispatch();
        });
    }

    function renderDash() {
        const grid = document.getElementById('dash-grid');
        grid.innerHTML = '';
        for (let i=1; i<=4; i++) {
            const c = db.courts[i];
            grid.innerHTML += `
                <div class="monitor-card">
                    <div class="m-header"><span>COURT ${i}</span> <span class="tag">${c.game}</span></div>
                    <div class="m-score">${c.s1} - ${c.s2}</div>
                    <div class="m-teams">${c.n1} vs ${c.n2}</div>
                    <a href="?view=scorer&court=${i}" target="_blank" class="btn btn-primary" style="justify-content:center"><i class="ph ph-game-controller"></i> Open Scorer</a>
                </div>
            `;
        }
    }

    function renderTeams() {
        const tbody = document.getElementById('team-list');
        tbody.innerHTML = '';
        Object.values(db.teams).forEach(t => {
            tbody.innerHTML += `
                <tr>
                    <td><img src="${t.logo}" class="team-logo-small"></td>
                    <td><b>${t.name}</b></td>
                    <td>${t.p1} & ${t.p2}</td>
                    <td><button class="btn btn-danger" onclick="delTeam('${t.id}')">Delete</button></td>
                </tr>
            `;
        });
    }

    function renderDispatch() {
        const mkOpt = (t) => `<option value="${t.id}">${t.name}</option>`;
        const opts = Object.values(db.teams).map(mkOpt).join('');
        document.getElementById('disp-t1').innerHTML = opts;
        document.getElementById('disp-t2').innerHTML = opts;
    }

    // Actions
    function addTeam() {
        let fd = new FormData();
        fd.append('action', 'save_team');
        fd.append('name', document.getElementById('new-t-name').value);
        fd.append('p1', document.getElementById('new-t-p1').value);
        fd.append('p2', document.getElementById('new-t-p2').value);
        fd.append('logo', document.getElementById('new-t-logo').value);
        fetch('', {method:'POST', body:fd}).then(loadDB);
    }

    function delTeam(id) {
        if(!confirm('Delete team?')) return;
        let fd = new FormData();
        fd.append('action', 'delete_team');
        fd.append('id', id);
        fetch('', {method:'POST', body:fd}).then(loadDB);
    }

    function dispatchMatch() {
        let fd = new FormData();
        fd.append('action', 'dispatch_match');
        fd.append('court_id', document.getElementById('disp-court').value);
        fd.append('t1_id', document.getElementById('disp-t1').value);
        fd.append('t2_id', document.getElementById('disp-t2').value);
        fd.append('match_title', document.getElementById('disp-title').value);
        fetch('', {method:'POST', body:fd}).then(() => {
            alert('Match Sent to Court!');
            loadDB();
        });
    }

    loadDB();
    setInterval(loadDB, 5000); // Auto refresh dashboard
</script>

<?php elseif($view == 'scorer'): ?>
<div class="scorer-body">
    <div class="scr-header">
        <div style="font-weight:bold">COURT <?php echo $courtID; ?></div>
        <div id="scr-status" style="font-size:12px; opacity:0.8">Live</div>
    </div>
    
    <div class="scr-grid">
        <div class="scr-team">
            <div class="scr-name" id="n1">Team A</div>
            <div class="scr-score" id="s1">0</div>
            <button class="scr-srv-btn" id="srv1" onclick="setServ(1)">SERVE HERE</button>
            <div class="scr-ctrl">
                <button class="btn-big sub" onclick="mod(1,-1)">-</button>
                <button class="btn-big add" onclick="mod(1,1)">+ Point</button>
            </div>
        </div>
        <div class="scr-team">
            <div class="scr-name" id="n2">Team B</div>
            <div class="scr-score" id="s2">0</div>
            <button class="scr-srv-btn" id="srv2" onclick="setServ(2)">SERVE HERE</button>
            <div class="scr-ctrl">
                <button class="btn-big sub" onclick="mod(2,-1)">-</button>
                <button class="btn-big add" onclick="mod(2,1)">+ Point</button>
            </div>
        </div>
    </div>

    <div class="scr-footer">
        <button class="btn btn-accent" style="font-size:18px" onclick="sideOut()">SIDE OUT ⇄</button>
        <div class="monitor-card" style="align-items:center; justify-content:center; background:#1e293b; color:white; border:none;" onclick="togSrv()">
            <div style="font-size:10px; text-transform:uppercase; color:#94a3b8">Server</div>
            <div style="font-size:24px; font-weight:bold" id="srv-num">2</div>
        </div>
    </div>
</div>
<script>
    let court = <?php echo $courtID; ?>;
    let state = {};

    function load() {
        fetch('?api_read=1&t='+Date.now()).then(r=>r.json()).then(d => {
            state = d.courts[court];
            render();
        });
    }

    function render() {
        document.getElementById('n1').innerText = state.n1;
        document.getElementById('n2').innerText = state.n2;
        document.getElementById('s1').innerText = state.s1;
        document.getElementById('s2').innerText = state.s2;
        document.getElementById('srv-num').innerText = state.serv;
        
        document.getElementById('srv1').className = state.team==1 ? 'scr-srv-btn active' : 'scr-srv-btn';
        document.getElementById('srv2').className = state.team==2 ? 'scr-srv-btn active' : 'scr-srv-btn';
    }

    function sync() {
        render(); // Optimistic UI
        let fd = new FormData();
        fd.append('action', 'update_score');
        fd.append('court', court);
        fd.append('data', JSON.stringify(state));
        fetch('', {method:'POST', body:fd});
    }

    function mod(t, v) { 
        if(t==1) state.s1 = Math.max(0, state.s1+v);
        if(t==2) state.s2 = Math.max(0, state.s2+v);
        sync();
    }
    function setServ(t) { state.team=t; sync(); }
    function togSrv() { state.serv = state.serv==1?2:1; sync(); }
    function sideOut() { state.team = state.team==1?2:1; state.serv=1; sync(); }

    load();
    setInterval(load, 3000); // Polling for name changes from dispatcher
</script>

<?php elseif($view == 'overlay'): ?>
<div id="overlay">
    <div class="ov-board">
        <div class="ov-team" id="ot1" style="flex-direction:row-reverse; text-align:right">
            <div class="ov-dot" style="right:15px"></div>
            <img id="ol1" class="ov-logo">
            <div class="ov-info">
                <div class="ov-name" id="on1"></div>
                <div class="ov-players" id="op1"></div>
            </div>
            <div class="ov-score" id="os1"></div>
        </div>
        <div class="ov-mid">
            <div style="font-size:11px; text-transform:uppercase; color:#94a3b8; margin-bottom:5px" id="ogame"></div>
            <div class="ov-srv-num" id="osrv"></div>
            <div style="font-size:9px; color:#64748b; margin-top:5px">SERVER</div>
        </div>
        <div class="ov-team" id="ot2">
            <div class="ov-dot" style="left:15px"></div>
            <div class="ov-score" id="os2"></div>
            <div class="ov-info">
                <div class="ov-name" id="on2"></div>
                <div class="ov-players" id="op2"></div>
            </div>
            <img id="ol2" class="ov-logo">
        </div>
    </div>
</div>
<script>
    let court = <?php echo $courtID; ?>;
    function load() {
        fetch('?api_read=1&t='+Date.now()).then(r=>r.json()).then(d => {
            const s = d.courts[court];
            document.getElementById('on1').innerText = s.n1;
            document.getElementById('on2').innerText = s.n2;
            document.getElementById('os1').innerText = s.s1;
            document.getElementById('os2').innerText = s.s2;
            document.getElementById('op1').innerText = s.p1 || '';
            document.getElementById('op2').innerText = s.p2 || '';
            document.getElementById('osrv').innerText = s.serv;
            document.getElementById('ogame').innerText = s.game;
            
            document.getElementById('ol1').src = s.l1;
            document.getElementById('ol2').src = s.l2;

            document.getElementById('ot1').classList.remove('serving');
            document.getElementById('ot2').classList.remove('serving');
            if(s.team==1) document.getElementById('ot1').classList.add('serving');
            else document.getElementById('ot2').classList.add('serving');
        });
    }
    setInterval(load, 1000);
</script>
<?php endif; ?>
</body>
</html>